<?php

namespace App\Services\Apps;

use App\Models\SocialChannel;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;

class FacebookApp
{
    private static string $appId;
    private static string $appSecret;
    public const MESSENGERKEY = 857909;

    public function __construct()
    {
        self::$appId = env('FB_APP_ID');
        self::$appSecret = env('FB_APP_SECRET');
    }

    private static function getRedirectUri(string $subdomain): string
    {

        return url('/api/callback') . '?state=' . $subdomain;
    }


    /**
     * Build Facebook login URL with permissions
     */
    public function login(string $subdomain): string
    {
        $permissions = [
            'pages_show_list',
            'business_management',
            'pages_messaging',
            'instagram_basic',
            'instagram_manage_messages',
            'pages_manage_metadata',

        ];

        $query = http_build_query([
            'client_id' => self::$appId,
            'redirect_uri' => self::getRedirectUri($subdomain),
            'scope' => implode(',', $permissions),
            'response_type' => 'code',
        ]);

        return "https://www.facebook.com/v22.0/dialog/oauth?{$query}";
    }

    /**
     * Process Facebook OAuth callback
     */
    public function callback(string $subdomain, $subscription)
    {
        $code = request()->query('code');

        if (!$code) {
            abort(401, 'No code received');
        }

        $tokenUrl = 'https://graph.facebook.com/v22.0/oauth/access_token?' . http_build_query([
            'client_id' => self::$appId,
            'redirect_uri' => self::getRedirectUri($subdomain),
            'client_secret' => self::$appSecret,
            'code' => $code,
        ]);

        $response = Http::timeout(10)->get($tokenUrl);

        if (!$response->ok()) {
            abort(500, 'Failed to get access token');
        }

        $data = $response->json();

        if (!isset($data['access_token'])) {
            abort(500, 'Failed to get access token');
        }

        $accessToken = $data['access_token'];

        // Get pages info
        $pagesUrl = "https://graph.facebook.com/v22.0/me/accounts?access_token={$accessToken}";
        $pagesResponse = Http::timeout(10)->get($pagesUrl);

        if (!$pagesResponse->ok()) {
            abort(500, 'Failed to get pages data');
        }



        $pagesData = $pagesResponse->json();
        $result = [];

        if (!empty($pagesData['data'])) {
            foreach ($pagesData['data'] as $page) {
                $pageId = $page['id'];
                $pageAccessToken = $page['access_token'];
                $webhookEvents = [
                    'inbox_labels',
                    'message_deliveries',
                    'message_echoes',
                    'message_reactions',
                    'message_reads',
                    'messages',
                    'messaging_account_linking',
                    'messaging_handovers',
                    'messaging_optins',
                    'messaging_policy_enforcement',
                    'messaging_postbacks',
                    'messaging_referrals',
                ];

                $subscribeUrl = "https://graph.facebook.com/v22.0/{$pageId}/subscribed_apps";
                $subscribeResponse = Http::timeout(10)->post($subscribeUrl, [
                    'access_token' => $pageAccessToken,
                    'subscribed_fields' => implode(',', $webhookEvents),
                ]);

                $igUrl = "https://graph.facebook.com/v22.0/{$pageId}?fields=instagram_business_account&access_token={$pageAccessToken}";
                $igResponse = Http::timeout(10)->get($igUrl);
                $igData = $igResponse->json();
                if (!empty($igData['instagram_business_account']['id'])) {
                    $igId = $igData['instagram_business_account']['id'];

                    $igSubscribeUrl = "https://graph.facebook.com/v22.0/{$igId}/subscribed_apps";
                    $igSubscribeResponse = Http::timeout(10)->post($igSubscribeUrl, [
                        'access_token' => $pageAccessToken,
                        'subscribed_fields' => 'messages',
                    ]);

                    if (!$igSubscribeResponse->ok()) {
                        Log::error("Failed to subscribe Instagram account to webhook: {$igId} - " . $igSubscribeResponse->body());
                    }
                }

                $result[] = [
                    'page_name' => $page['name'] ?? null,
                    'page_id' => $pageId,
                    'page_token' => $pageAccessToken,
                    'instagram_id' => $igData['instagram_business_account']['id'] ?? null,
                ];
            }
        }


        $this->syncPagesToServer($subscription, $result);
        return true;
    }

    public function removePageFromApp(string $pageId, string $pageAccessToken, ?string $igId = null): bool
    {
        // 1. Hapus webhook subscription dari Page
        $unsubscribeUrl = "https://graph.facebook.com/v22.0/{$pageId}/subscribed_apps";
        $unsubscribeResponse = Http::timeout(10)->delete($unsubscribeUrl, [
            'access_token' => $pageAccessToken,
        ]);

        if (!$unsubscribeResponse->ok()) {
            Log::error("Failed to unsubscribe app from page {$pageId}: " . $unsubscribeResponse->body());
            return false;
        }

        // 2. Jika Instagram ID tersedia, hapus juga webhook subscription dari akun IG
        if (!empty($igId)) {
            $igUnsubscribeUrl = "https://graph.facebook.com/v22.0/{$igId}/subscribed_apps";
            $igUnsubscribeResponse = Http::timeout(10)->delete($igUnsubscribeUrl, [
                'access_token' => $pageAccessToken,
            ]);

            if (!$igUnsubscribeResponse->ok()) {
                Log::error("Failed to unsubscribe app from Instagram account {$igId}: " . $igUnsubscribeResponse->body());
            }
        }

        // 3. Hapus permission app dari Page
        $permissionsUrl = "https://graph.facebook.com/v22.0/{$pageId}/permissions";
        $permissionsResponse = Http::timeout(10)->delete($permissionsUrl, [
            'access_token' => $pageAccessToken,
        ]);

        if (!$permissionsResponse->ok()) {
            Log::error("Failed to remove app permission from page {$pageId}: " . $permissionsResponse->body());
            return false;
        }

        return true;
    }



    public function syncPagesToServer($subscription, $pages)
    {


        $serverHost = $subscription->server->host;
        $secretKey = self::MESSENGERKEY;
        $protocol = env('APP_ENV') === "local" ? 'http://' : 'https://';
        $responseKey = Http::withoutVerifying()->post($protocol . $serverHost . '/save-messenger-key', [
            'secretKey' => $secretKey,
            'subdomain' => $subscription->domain
        ]);

        if (!$responseKey->ok()) {
            throw new \Exception('Failed to send messenger key');
        }

        foreach ($pages as $page) {
            $response = Http::withoutVerifying()->post($protocol . $serverHost . '/save-setting-messenger', [
                'page_name' => $page['page_name'],
                'page_id' => $page['page_id'],
                'page_token' => $page['page_token'],
                'instagram_id' => $page['instagram_id'],
                'subdomain' => $subscription->domain,
            ]);

            if ($response->ok()) {
                SocialChannel::updateOrCreate(
                    [
                        'subscription_id' => $subscription->id,
                        'channel_id' => $page['page_id'],
                        'platform' => 'messenger',
                    ],
                    [
                        'channel_type' => 'page',
                        'channel_name' => $page['page_name'],
                        'access_token' => $page['page_token'],
                        'instagram_id' => $page['instagram_id'],
                        'connected_at' => now(),
                        'webhook_verified_at' => now(),
                    ]
                );
            } else {
                Log::error('Failed to save page settings for page_id: ' . $page['page_id']);
            }
        }
    }
}
