<?php

namespace App\Notifications;

use App\Clients\WaNotifClient;
use App\Libs\Helper;
use App\Notifications\Channels\WaChannel;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Notifications\Messages\MailMessage;
use Illuminate\Notifications\Notification;

class SubscriptionActive extends Notification implements ShouldQueue
{
    use Queueable;

    protected $subscription;
    protected $guideLines;

    /**
     * Create a new notification instance.
     */
    public function __construct($subscription)
    {
        $this->subscription = $subscription;

        // Cek apakah domain sudah ada atau belum, dan siapkan panduan untuk pelanggan
        $isDomain = Helper::isDomain($subscription->domain);
        if ($isDomain) {
            try {
                $ns = explode(',', $subscription->ns);
                $this->guideLines = [
                    "Silahkan setting nameserver domain anda ke:",
                    $ns[0],
                    $ns[1]
                ];
            } catch (\Throwable $th) {
                $this->guideLines = [
                    "Silahkan cek name server di halaman langganan dan hubungkan dengan domain anda"
                ];
            }
        } else {
            $doc = env("DOC_URL");
            $detail = json_decode($subscription->details);
            $this->guideLines = [
                "Login akses : " . $subscription->domain,
                "Email: " . $detail->email,
                "Password: " . $detail->password,
                "Dokumentasi: " . $doc
            ];
        }
    }

    /**
     * Get the notification's delivery channels.
     *
     * @return array<int, string>
     */
    public function via(object $notifiable): array
    {
        return ['mail', WaChannel::class];
    }

    /**
     * Get the mail representation of the notification.
     */
    public function toMail(object $notifiable): MailMessage
    {
        $mailMessage = (new MailMessage)
            ->subject('Langganan Anda Telah Aktif')
            ->line("Langganan aktif")
            ->line("Halo " . $notifiable->name . ",")
            ->line("Langganan anda untuk domain " . $this->subscription->domain . " dengan produk " . $this->subscription->product . " telah aktif.");

        foreach ($this->guideLines as $line) {
            $mailMessage->line($line);
        }

        return $mailMessage
            ->line("Terima kasih telah menggunakan layanan kami.")
            ->line("Regards,")
            ->line(config('app.name'));
    }

    /**
     * Get the WhatsApp representation of the notification.
     */
    public function toWa(object $notifiable)
    {
        $waMessage = (new WaNotifClient())
            ->bold('Langganan aktif')->line()
            ->line("Halo " . $notifiable->name . ",")->line()
            ->line("Langganan anda untuk domain " . $this->subscription->domain . " dengan produk " . $this->subscription->product . " telah aktif.")
            ->line()->line();
        foreach ($this->guideLines as $line) {
            $waMessage->line($line);
        }

        return $waMessage->line()->line()
            ->line("Terima kasih telah menggunakan layanan kami.")
            ->line("Regards,")
            ->line(config('app.name'));
    }

    /**
     * Get the array representation of the notification.
     *
     * @return array<string, mixed>
     */
    public function toArray(object $notifiable): array
    {
        return [
            // ...
        ];
    }
}
