<?php

namespace App\Jobs;

use App\Clients\AapanelApi;
use App\Clients\CloudflareApi;
use App\Clients\ConfigWebsiteSsh;
use App\Clients\WaNotifClient;
use App\Models\Payment;
use App\Models\Subscription;
use Illuminate\Support\Facades\Log;
use App\Helpers\Helper;
use App\Libs\Helper as LibsHelper;
use App\Models\Server;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use Throwable;

class UpdateSubscriptionStatus implements ShouldQueue
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    protected $payment;
    private $adminNo;
    private $notif;
    public function __construct(Payment $payment)
    {
        $this->payment = $payment;
        $this->notif = new WaNotifClient();
        $this->adminNo = env('NO_ADMIN');
    }

    public function handle()
    {
        try {
            $payment = $this->payment;
            $user = $payment->user;

            $subscription  = $payment->subscription;

            if ($payment->payment_type === "initial") {

                $subscription->start_date = now();
                $domain = $subscription->domain;
                $isDomain = LibsHelper::isDomain($domain);

                if (!$isDomain) {
                    $detail = $this->createWebsite($domain, $user);
                    $subscription->server_id = $detail['server_id'];
                    $subscription->details = json_encode($detail['detail']);
                    $subscription->status = $detail['status'];
                    $subscription->web_id = $detail['id'];
                    if ($detail['status'] === 'processing') {

                        $l = $this->notif->line('Ada error saat pembuatan website')->line()
                            ->line('Domain ' . $domain)
                            ->line('Product' . $subscription->product)->line()->line();
                        foreach ($detail['errors'] as $error) {
                            Log::info($error);
                            $l->line($error);
                        }

                        $l->line();
                        $l->line()->to($this->adminNo)->send();
                    }
                } else {
                    $subscription->status = 'processing';
                    $subscription->ns = $this->getNameServer($domain);
                    $this->notif->line('Pembuatan website perlu di proses')->line()
                        ->line('Domain ' . $domain)
                        ->line('Product' . $subscription->product)
                        ->to($this->adminNo)->send();
                }

                $subscription->save();
            } else if ($payment->payment_type === "renewal") {
                if ($subscription->status !== 'active') {
                    $subscription->status = 'active';
                }
                $subscription->due_date = now()->addMonth();
                $subscription->save();
            }
        } catch (Throwable $th) {
            Log::error($th->getMessage());
        }
    }

    protected function createWebsite($subdomain, $user)
    {
        $isSuccess = true;
        $errors = [];

        $server = Server::where('is_open', 1)->inRandomOrder()->first();
        if (!$server) {
            $isSuccess = false;
            $errors[] = 'Server not found (db)';
        }


        $cloudflare = new CloudflareApi();
        $aapanel = new AapanelApi($server->panel_url, $server->panel_api_key);
        $configWeb = new ConfigWebsiteSsh($server);

        $createDns = $cloudflare->createSubdomain($subdomain, $server->host);
        if (!$createDns) {
            $isSuccess = false;
            $errors[] = 'Error creating subdomain (add dns) CloudflareApi ' . $subdomain;
        }
        sleep(10);
        $somechar = substr(str_replace('.', '', $subdomain), 0, 7);
        $userdb = $somechar . '_ilive';
        $passdb = rand(100000, 999999) . 'aA';
        $ssl = false;
        try {
            $createWebsite = $aapanel->createWebsite($subdomain, $subdomain, $userdb, $passdb);
            $aapanel->addHeadersConfigToConfig($subdomain);
            $ssl = $createWebsite['ssl'] ?? false;
        } catch (\Throwable $th) {
            $isSuccess = false;
            $errors[] = 'Error creating website AapanelApi , server : *' . $server->host . '*' . "|" . $th->getMessage();
        }
        $detail = [];
        try {

            $detail = $configWeb->connect()
                ->uploadAndExtractFile($subdomain)
                ->config($userdb, $passdb, $userdb, $user, $ssl);
        } catch (\Throwable $th) {
            $isSuccess = false;
            $errors[] = 'Error upload folder and config install ConfigWebsiteSsh ' . $th->getMessage();
        }
        $webId = null;
        try {
            $getWeblist = $aapanel->getWebsiteList($subdomain, 1);

            if (!empty($getWeblist['data']) && isset($getWeblist['data'][0]['id'])) {
                $webId = (int) $getWeblist['data'][0]['id'];
            } else {

                throw new \Exception("Website tidak ditemukan untuk subdomain: $subdomain");
            }
        } catch (\Exception $e) {
            report($e);
        }

        return [
            'status' => $isSuccess ? 'active' : 'processing',
            'errors' => $errors,
            'detail' => $detail,
            'id' => $webId,
            'server_id' => $server->id,

        ];
    }


    protected function getNameServer($domain)
    {
        $cloudflare = new CloudflareApi();
        $createDns = $cloudflare->inputDomain($domain);

        $nameservers = $createDns['result']['name_servers'] ?? [];  // Defaultkan ke array kosong jika null

        if (!empty($nameservers)) {
            return implode(',', $nameservers);
        } else {
            // if name_server empty or null
            Log::error("belum di tetapkan");
            return '';  // Atau return nilai default lainnya
        }
    }
}
