<?php

namespace App\Http\Controllers;

use App\Http\Requests\OrderRequest;
use App\Models\Payment;
use App\Models\Product;
use App\Models\ProductPackage;
use App\Models\Subscription;
use App\Models\User;
use App\Notifications\NewInvoice;
use Illuminate\Container\Attributes\Auth;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth as FacadesAuth;
use Illuminate\Support\Facades\DB;

class OrderController extends Controller
{
    public function form($productSlug, $productPackageId)
    {
        $product = Product::where('slug', $productSlug)->orderBy('id', 'desc')->firstOrFail();
        $package = ProductPackage::findOrFail($productPackageId);

        return view('orders.form', compact('product', 'package'));
    }

    public function store(OrderRequest $request)
    {

        try {
            $product = Product::findOrFail($request->product_id);
            $package = ProductPackage::findOrFail($request->package_id);
            $user = FacadesAuth::user() ?? null;

            if (!FacadesAuth::check()) {
                $user = User::create([
                    'name' => $request->name,
                    'email' => $request->email,
                    'password' => bcrypt($request->password),
                    'whatsapp' => $request->whatsapp,
                ]);
                FacadesAuth::login($user);
            }
            //
            if ($package->price === 0 && $user->trial)  return $this->backError('Anda sudah tidak bisa menggunakan trial');
            DB::BeginTransaction();
            $subscription = Subscription::create([
                'user_id' => $user->id,
                'product' => $product->name,
                'price' => $package->price,
                'domain' => $request->domain_option == 'own' ? $request->domain : $request->domain . '.' . env('APP_DOMAIN','interaksi.live'),
                'due_date' => now()->addDays($package->duration),
                'status' => 'waiting',

            ]);
            //
            $inv =  Payment::create([
                'invoice_number' => 'INV' . time() . '' . $user->id,
                'subscription_id' => $subscription->id,
                'description' => "Tagihan pertama untuk langganan " . $product->name . " " . $package->name . " " . ($subscription->domain),
                'user_id' => $user->id,
                'amount' => $package->price,
                'expired_at' => now()->addDays(3),

            ]);
            $notif = (new NewInvoice($inv))->delay(now()->addSeconds(5));
            $user->notify($notif);
            DB::commit();
            return $this->redirectSuccess(route('invoices.payment', $inv->invoice_number), 'Pesanan berhasil dibuat,anda dapat melakukan pembayaran');
        } catch (\Throwable $th) {
          
            DB::rollBack();
            return $this->backError();
        }
    }
}
