<?php

namespace App\Http\Controllers;

use App\Clients\DuitkuClient;
use App\Clients\IpaymuService;
use App\Models\Payment;
use App\Models\PaymentMethod;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;

class InvoiceController extends Controller
{
    public function index(Request $request)
    {
        $invoices = $request->user()->payments()->orderBy('created_at', 'desc')->get();
        return view('invoices.index', compact('invoices'));
    }

    public function detail(Payment $invoice, Request $request)
    {
        if (!$invoice->payment_details)     return redirect(route('invoices.payment', $invoice->invoice_number));
        abort_if($invoice->user_id !== $request->user()->id && !$request->user()->is_admin, 404);
        $invoice->load('user');
        return view('invoices.detail', compact('invoice'));
    }

    public function payment(Payment $invoice)
    {
        if ($invoice->payment_details) return redirect(route('invoices.detail', $invoice->invoice_number));

        $paymentMethods = PaymentMethod::all();
        return view('invoices.payment', compact('invoice', 'paymentMethods'));
    }

    public function processPayment(Request $request)
    {
        try {
            $user = $request->user();
            $validated = $request->validate([
                'payment_method' => 'required|exists:payment_methods,code',
                'discount' => 'nullable',
                'invoice_id' => 'required|exists:payments,id',
            ]);
            $method = PaymentMethod::where('code', $validated['payment_method'])->first();
            $payment = Payment::findOrFail($validated['invoice_id']);
            if ($payment->amount == 0) {
                if(!$method->is_manual){
                    return $this->backError("Pilih metode manual untuk trial.");
                }
                $user->update(['trial' => true]);
            }
            $paymentDetails = $this->getDetailPayment($payment, $method, $user);

            $payment->update([
                'payment_details' => $paymentDetails['details'],
                'tax' => $paymentDetails['fee'],
                'status' => $payment->amount == 0 ? 'paid' : 'unpaid',
                'is_manual' => $method->is_manual,
            ]);


            return redirect(route('invoices.detail', $payment->invoice_number));
        } catch (\Throwable $th) {
            throw $th;
            Log::error($th->getMessage());
            return $this->backError();
        }
    }


    private function getDetailPayment($invoice, $method, $user)
    {
        if ($method->is_manual) {
            $details = json_encode([
                'type' => $method->type,
                'bank_name' => $method->name,
                'bank_account' => $method->bank_account,
                'bank_account_name' => $method->bank_account_name,
                'qr_code' => null,
                'expired' => null,

            ]);
            $fee = $method->fee;
            return [
                'details' => $details,
                'fee' => $fee,
            ];
        }

         if($method->provider === 'duitku') {
            $duitku = new DuitkuClient();
            $pay = $duitku->createTransaction($invoice->invoice_number, $invoice->amount, $invoice->description, $user->email);

            $pay['type'] = $method->type;
            $details = json_encode($pay);
            $fee = 0;


        } else if ($method->provider === "ipaymu"){
            $fee = 0;
            $ipaymu = new IpaymuService();
            $pay = $ipaymu->createTransaction($user,$invoice->invoice_number,$method,$invoice->amount,$invoice->description);
            $data = $pay['Data'];
            $paymentNo = $data['PaymentNo'];
            $isQr = !ctype_digit($paymentNo);

            $details = json_encode([
                'type' => $method->type,
                'amount' => $data['Total'],
                'bank_account_name' => $data['PaymentName'],
                'bank_account' => $isQr ? null : $paymentNo,
                'qr_code' => $isQr ? $paymentNo : null,
                'expired' => $data['Expired']
            ]);
        }

        return [
            'details' => $details,
            'fee' => $fee,
        ];


    }
}
