<?php

namespace App\Http\Controllers\Admin;

use App\Clients\AapanelApi;
use App\Clients\ConfigWebsiteSsh;
use App\Http\Controllers\Controller;
use App\Models\Server;
use Error;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;
use Yajra\DataTables\DataTables;

class ServerController extends Controller
{

    public function index(Request $request)
    {
        if ($request->ajax()) {
            $data = Server::all();
            return DataTables::of($data)
                ->addIndexColumn()
                ->addColumn('action', function ($row) {
                    $editUrl = route('admin.servers.edit', $row->id);
                    $detailUrl = route('admin.servers.detail', $row->id);
                    $btn = '<a href="' . $editUrl . '" class="btn btn-warning btn-sm">Edit</a>';
                    $btn .= '<button data-url="' . route('admin.servers.destroy', $row->id) . '" class="btn btn-danger btn-sm delete ml-2">Delete</button>';
                    $btn .= '<a href="' . $detailUrl . '" class="btn btn-info btn-sm">Detail</a>';
                    return $btn;
                })
                ->rawColumns(['action'])
                ->make(true);
        }
        return view('admin.servers.index');
    }

    public function create()
    {
        return view('admin.servers.create');
    }

    public function store(Request $request)
    {
        $this->validateServer($request);
        Server::create($request->all());
        return $this->redirectSuccess(route('admin.servers.index'), 'Server created successfully.');
    }

    public function edit($id)
    {
        $server = Server::findOrFail($id);
        return view('admin.servers.edit', compact('server'));
    }

    public function update(Request $request, $id)
    {
        $this->validateServer($request);
        $server = Server::findOrFail($id);
        $server->update($request->all());
        return $this->redirectSuccess(route('admin.servers.index'), 'Server updated successfully.');
    }

    protected function validateServer(Request $request)
    {
        return $request->validate([
            'host' => 'required|string|max:255',
            'panel_url' => 'required|string|max:255',
            'username' => 'required|string|max:255',
            'sshkey' => 'required|string',
            'is_open' => 'required|boolean',
            'port' => 'required|integer|min:1|max:65535',
            'dbuser' => 'nullable|string|max:255',
            'dbpass' => 'nullable|string|max:255',
            'panel_api_key' => 'nullable|string|max:255',
        ]);
    }

    public function destroy(Server $server)
    {
        $server->delete();
        return response()->json(['message' => 'Server deleted successfully.']);
    }

    public function checkConnection(Request $request)
    {
        try {
            $result = null;
            $type = $request->type;
            if ($type === "api") {
                $apiUrl = $request->aapanelUrl;
                $key = $request->aapanelKey;
                $aapanel = new AapanelApi($apiUrl, $key);
                $result = $aapanel->getLogs();
            } else if ($type === "ssh") {
                $fakeServer = new Server([
                    'host' => $request->host,
                    'port' => $request->port,
                    'username' => $request->username,
                    'sshkey' => $request->sshKey,
                ]);
                $config = new ConfigWebsiteSsh($fakeServer);
                $result = ['data' => $config->connect()->getServerStats()];
                $config->disconnect();
            }
            if (!$result) return ['status' => false, 'msg' => "something went wrong,please recheck data"];
            return $result;
        } catch (\Throwable $th) {
            return json_encode(['error' => "something went wrong"]);
        }
    }


    public function detailServer(Request $request, Server $server)
    {
        $aapanel = new AapanelApi($server->panel_url, $server->panel_api_key);

        if ($request->ajax()) {
            $result = $aapanel->getWebsitelist();
            if (!isset($result['data']) || !is_array($result['data'])) {
                return response()->json([
                    'data' => [],
                    'recordsTotal' => 0,
                    'recordsFiltered' => 0,
                    'error' => 'Gagal mengambil data dari Aapanel.'
                ]);
            }

            $data = $result['data'];
            $total = 0;
            Log::info(json_encode($data));

            // Coba ambil total dari HTML string di field 'page'
            if (isset($result['page'])) {
                preg_match("/Total (\d+)/", $result['page'], $matches);
                $total = isset($matches[1]) ? (int)$matches[1] : count($data);
            }

            return DataTables::of($data)
                ->addIndexColumn()
                ->addColumn('name', fn($row) => $row['name'])
                ->addColumn('ssl_expiry', function ($row) {
                    return isset($row['ssl']['notAfter'])
                        ? \Carbon\Carbon::parse($row['ssl']['notAfter'])->format('d M Y')
                        : '-';
                })
                ->addColumn('status', function ($row) {
                    return $row['status'] == '1'
                        ? '<span class="badge bg-success">Running</span>'
                        : '<span class="badge bg-secondary">Stopped</span>';
                })
                ->addColumn('addtime', fn($row) => \Carbon\Carbon::parse($row['addtime'])->format('d M Y H:i'))

                ->rawColumns(['status'])
                ->setTotalRecords($total)
                ->setFilteredRecords($total)
                ->skipPaging()
                ->make(true);
        }
        $logs = $aapanel->getLogs();
        return view('admin.servers.detail', ['server' => $server, 'logs' => $logs]);
    }
}
