<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Product;
use Illuminate\Http\Request;
use Yajra\DataTables\DataTables;

class ProductController extends Controller
{

    public function index(Request $request)
    {
        if ($request->ajax()) {
            $data = Product::withCount('packages');
            return DataTables::of($data)
                ->addIndexColumn()
                ->addColumn('packages_count', function ($row) {
                    return $row->packages_count;
                })->addColumn('image', function ($row) {
                    //if is url return string if path return image
                    return strpos($row->image, 'http') === false ? asset($row->image) : $row->image;
                })
                ->addColumn('action', function ($row) {
                    $href = route('admin.products.edit', $row->id);
                    $btn = '<a href="' . $href . '"  data-id="' . $row->id . '"   class="edit ml-2 btn btn-danger btn-sm">Edit</a>';
                    return $btn;
                })
                ->rawColumns(['action'])
                ->make(true);
        }
        return view('admin.products.index');
    }

    public function create()
    {
        return view('admin.products.create');
    }

    public function edit(Product $product)
    {
        return view('admin.products.edit', compact('product'));
    }

    public function store(Request $request)
    {
        $this->validateRequest($request);

        // Buat produk baru
        $product = new Product();
        $product = $this->createOrUpdateProduct($product, $request);
        return $this->redirectSuccess(route('admin.products.index'), 'Product created successfully.');
    }


    // Memperbarui produk yang sudah ada
    public function update(Request $request, $id)
    {
        $this->validateRequest($request, $id);
        // Temukan produk
        $product = Product::findOrFail($id);
        $product = $this->createOrUpdateProduct($product, $request);
        return $this->redirectSuccess(route('admin.products.index'), 'Product updated successfully.');
    }

    private function createOrUpdateProduct(Product $product, Request $request)
    {
        $product->code = $request->code;
        $product->name = $request->name;
        $product->slug = $request->slug;
        $product->description = $request->description;
        $product->avaibility = $request->avaibility;

        if ($request->hasFile('image')) {
            $product->image = $this->handleImageUpload($request->image);
        }

        $product->save();

        return $product;
    }

    private function handleImageUpload($image)
    {
        $imageName = time() . '.' . $image->extension();
        $image->move(public_path('images/product'), $imageName);
        return 'images/product/' . $imageName;
    }

    // Validasi request
    protected function validateRequest(Request $request, $id = null)
    {
        $rules = [
            'code' => 'required|string|max:255',
            'name' => 'required|string|max:255',
            'slug' => 'required|string|max:255|unique:products,slug,' . $id,
            'description' => 'required|string',
            'avaibility' => 'required|boolean',
            'image' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048',
        ];

        return $request->validate($rules);
    }
}
