<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\PaymentMethod;
use Illuminate\Http\Request;
use Yajra\DataTables\Facades\DataTables;

class PaymentChannelController extends Controller
{
    public function index(Request $request)
    {
        if ($request->ajax()) {
            $data = PaymentMethod::query();

            return DataTables::of($data)
                ->addIndexColumn()
                ->addColumn('type', fn($row) => ucfirst($row->type))
                ->addColumn('provider', fn($row) => ucfirst($row->provider) ?? "-")
                ->addColumn('manual', function ($row) {
                    return $row->is_manual
                        ? '<span class="badge bg-warning text-dark">Manual</span>'
                        : '<span class="badge bg-info text-dark">Otomatis</span>';
                })
                ->addColumn('status', function ($row) {
                    $badge = $row->is_active
                        ? '<span class="badge bg-success">Aktif</span>'
                        : '<span class="badge bg-secondary">Tidak Aktif</span>';

                    $toggle = '<button class="btn btn-sm btn-outline-' . ($row->is_active ? 'danger' : 'success') . ' toggle-status"
                            data-id="' . $row->id . '">'
                        . ($row->is_active ? 'Nonaktifkan' : 'Aktifkan') . '</button>';

                    return $badge . '<br>' . $toggle;
                })

                ->addColumn('image', function ($row) {
                    return $row->image
                        ? '<img loading="lazy" src="' .  $row->image . '" class="img-thumbnail" style="height: 40px;">'
                        : '-';
                })

                ->addColumn('action', function ($row) {

                    $deleteUrl = route('admin.payment_methods.destroy', $row->id);
                    return '

                    <button class="btn btn-sm btn-danger delete" data-url="' . $deleteUrl . '">Hapus</button>
                ';
                })
                ->rawColumns(['manual', 'status', 'image', 'action'])
                ->make(true);
        }
        return view('admin.payment-methods.index');
    }

    public function create()
    {
        return view('admin.payment-methods.create');
    }

    public function toggle($id)
    {
        $method = PaymentMethod::findOrFail($id);
        $method->is_active = !$method->is_active;
        $method->save();
        return json_encode(['success' => true]);
    }


    public function store(Request $request)
    {
        $validated = $request->validate([
            'code' => 'required|string|max:255|unique:payment_methods,code',
            'name' => 'required|string|max:255',
            'type' => 'required|in:VA,QRIS,Bank Transfer,EWALLET,RETAIL,OVO',

            'image' => 'nullable',
            'provider' => 'required|in:ipaymu,duitku,manual',
            'is_manual' => 'required|boolean',
            'is_active' => 'required|boolean',
            'bank_account_name' => 'nullable|string|max:255',
            'bank_account' => 'nullable|string|max:255',
        ]);

        if ($validated['provider'] === 'manual') {
            if (empty($validated['bank_account']) || empty($validated['bank_account_name'])) {
                return back()->withErrors([
                    'bank_account' => 'Bank account wajib diisi jika provider manual.',
                    'bank_account_name' => 'Bank account name wajib diisi jika provider manual.',
                ]);
            }
        } else {

            $validated['bank_account'] = null;
            $validated['bank_account_name'] = null;
        }

        $validated['provider'] = $validated['provider'] === 'manual' ? null : $validated['provider'];

        PaymentMethod::create($validated);

        return $this->redirectSuccess(route('admin.payment_methods.index'));
    }

    public function destroy($id)
    {
        PaymentMethod::whereId($id)->delete();
        return response()->json(['message' => 'Method deleted successfully.']);
    }
}
