<?php


namespace App\Http\Controllers\Admin;


use App\Http\Controllers\Controller;
use App\Models\Package;
use App\Models\Product;
use App\Models\ProductPackage;
use Illuminate\Http\Request;
use Yajra\DataTables\DataTables;

class PackageController extends Controller
{
    public function index(Request $request)
    {
        if ($request->ajax()) {
            $data = ProductPackage::with('product');
            return DataTables::of($data)
                ->addIndexColumn()
                ->addColumn('product', function ($row) {
                    return $row->product->name ?? 'N/A';
                })
                ->addColumn('action', function ($row) {
                    $editUrl = route('admin.packages.edit', $row->id);
                    $btn = '<a href="' . $editUrl . '" class="btn btn-warning btn-sm">Edit</a>';
                    $btn .= '<button data-url="' . route('admin.packages.destroy', $row->id) . '" class="btn btn-danger btn-sm delete ml-2">Delete</button>';
                    return $btn;
                })
                ->rawColumns(['action'])
                ->make(true);
        }
        return view('admin.packages.index');
    }

    public function create()
    {
        $products = Product::all();
        return view('admin.packages.create', compact('products'));
    }

    public function store(Request $request)
    {
        $this->validateRequest($request);

        ProductPackage::create($request->all());

        return redirect()->route('admin.packages.index')->with('success', 'Package created successfully.');
    }

    public function edit(ProductPackage $package)
    {
        $products = Product::all();
        return view('admin.packages.edit', compact('package', 'products'));
    }

    public function update(Request $request, ProductPackage $package)
    {
        $this->validateRequest($request, $package->id);
        $package->update($request->all());
        return $this->redirectSuccess(route('admin.packages.index'), 'Package updated successfully.');
    }

    protected function validateRequest(Request $request, $id = null)
    {
        $rules = [
            'product_id' => 'required|exists:products,id',
            'title' => 'required|string|max:255',
            'description' => 'required|string',
            'price' => 'required|integer',
            'external' => 'nullable|string',
            'duration' => 'required|integer|min:1',
        ];

        return $request->validate($rules);
    }

    public function destroy(ProductPackage $package)
    {
        $package->delete();
        return response()->json(['message' => 'Package deleted successfully.']);
    }
}
