<?php

namespace App\Clients;

use GuzzleHttp\Client;
use GuzzleHttp\Exception\RequestException;
use Illuminate\Support\Facades\Log;

class CloudflareApi
{
    protected $client;
    protected $apiKey;
    protected $email;
    protected $accountId;
    protected $zoneId;


    public function __construct()
    {
        $this->apiKey = env('CF_API_KEY', 'default_api_key');
        $this->accountId = env('CF_ACCOUNT_ID', 'default_account_id');
        $this->zoneId = env('CF_ZONE_ID', 'default_zone_id');


        $this->client = new Client([
            'base_uri' => 'https://api.cloudflare.com/client/v4/',
            'headers' => [
                'Content-Type' => 'application/json',
                'Authorization' => 'Bearer ' . $this->apiKey,
                'X-Auth-Key' => $this->apiKey,
            ],
        ]);
    }

    public function checkSubdomain($subdomain)
    {
        try {
            $response = $this->client->get("zones/{$this->zoneId}/dns_records", [
                'query' => [
                    'name' => $subdomain,
                    'type' => 'A',
                ],
            ]);
            $records = json_decode($response->getBody()->getContents(), true);
            if ($records['success'] && !empty($records['result'])) {
                return true;
            }
            return false;
        } catch (RequestException $e) {
            Log::error('Error checking subdomain in CloudflareApi: ' . $e->getMessage());
            return false;
        }
    }

    public function createSubdomain($subdomain, $ip)
    {
        if ($this->checkSubdomain($subdomain)) {
            return true;
        }

        try {
            $response = $this->client->post("zones/{$this->zoneId}/dns_records", [
                'json' => [
                    'type' => 'A',
                    'name' => $subdomain,
                    'content' => $ip,
                    'ttl' => 1,
                    'proxied' => false,
                ],
            ]);
            return true;
            //  return json_decode($response->getBody()->getContents(), true);
        } catch (RequestException $e) {
            Log::error('Error creating subdomain CloudflareApi: ' . $e->getMessage());
            return false;
            return [
                'success' => false,
                'message' => $e->getMessage(),
            ];
        }
    }

    public function inputDomain($domain)
    {
        try {
            $response = $this->client->post("zones", [
                'json' => [
                    'name' => $domain,
                    'account' => [
                        'id' => $this->accountId,
                    ],
                    'type' => 'full',
                ],
            ]);



            return json_decode($response->getBody()->getContents(), true);
        } catch (RequestException $e) {
            return [
                'success' => false,
                'message' => $e->getMessage(),
            ];
        }
    }

    public function deleteSubdomain($subdomain)
    {
        try {
            // Pertama, dapatkan ID DNS record subdomain
            $response = $this->client->get("zones/{$this->zoneId}/dns_records", [
                'query' => [
                    'name' => $subdomain,
                    'type' => 'A', // Ganti dengan jenis record yang sesuai jika perlu
                ],
            ]);

            $records = json_decode($response->getBody()->getContents(), true);
            if (!$records['success'] || empty($records['result'])) {
                return ['success' => false, 'message' => 'Subdomain not found'];
            }

            // Hapus DNS record subdomain
            $recordId = $records['result'][0]['id'];
            $response = $this->client->delete("zones/{$this->zoneId}/dns_records/{$recordId}");

            return json_decode($response->getBody()->getContents(), true);
        } catch (RequestException $e) {
            Log::error('Error deleting subdomain CloudflareApi: ' . $e->getMessage());
            return [
                'success' => false,
                'message' => $e->getMessage(),
            ];
        }
    }

    public function deleteDomain($domain)
    {
        try {
            //  ID zona untuk domain
            $response = $this->client->get("zones", [
                'query' => [
                    'name' => $domain,
                ],
            ]);

            $zones = json_decode($response->getBody()->getContents(), true);
            if (!$zones['success'] || empty($zones['result'])) {
                return ['success' => false, 'message' => 'Domain not found'];
            }

            // Ambil ID zona dari respons
            $zoneId = $zones['result'][0]['id'];

            // Hapus domain dengan menggunakan zone ID
            $response = $this->client->delete("zones/{$zoneId}");

            // Periksa apakah penghapusan berhasil
            $result = json_decode($response->getBody()->getContents(), true);
            if (!$result['success']) {
                return ['success' => false, 'message' => 'Failed to delete domain'];
            }

            return $result;
        } catch (RequestException $e) {
            Log::error('Error deleting domain CloudflareApi: ' . $e->getMessage());
            return [
                'success' => false,
                'message' => $e->getMessage(),
            ];
        }
    }



    public function getSubdomains($page = 1, $perPage = 100)
    {
        try {
            $domain = env('APP_DOMAIN');

            $zoneResponse = $this->client->get("zones", [
                'query' => [
                    'name' => $domain,
                    'status' => 'active',
                ],
            ]);

            $zoneData = json_decode($zoneResponse->getBody()->getContents(), true);
            if (!$zoneData['success'] || empty($zoneData['result'])) {
                return [
                    'success' => false,
                    'message' => 'Zone not found for domain: ' . $domain,
                ];
            }

            $zoneId = $zoneData['result'][0]['id'];

        
            $dnsResponse = $this->client->get("zones/{$zoneId}/dns_records", [
                'query' => [
                    'page' => $page,
                    'per_page' => $perPage,
                ],
            ]);

            $dnsData = json_decode($dnsResponse->getBody()->getContents(), true);
            if (!$dnsData['success']) {
                return [
                    'success' => false,
                    'message' => 'Failed to fetch DNS records',
                ];
            }

            $subdomains = collect($dnsData['result'])->filter(function ($record) use ($domain) {
                return
                    in_array($record['type'], ['A', 'CNAME']) &&
                    $record['name'] !== $domain && // Bukan domain utama
                    str_ends_with($record['name'], '.' . $domain); // Subdomain dari domain utama
            })->values()->all();

            return [
                'success' => true,
                'subdomains' => $subdomains,
                'total' => count($subdomains),
            ];
        } catch (RequestException $e) {
            Log::error('Error fetching subdomains from CloudflareApi: ' . $e->getMessage());
            return [
                'success' => false,
                'message' => $e->getMessage(),
            ];
        }
    }

}
