<?php

namespace App\Clients;

use Illuminate\Support\Facades\Log;

/**
 * aapanel API Interface Example
 * For reference only, please develop according to actual project requirements and handle security properly.
 * date: 2018/12/12
 * author: Aliang (Translated and modified)
 */
class AapanelApi
{
    private $API_KEY; // API key
    private $PANEL_URL; // Panel URL

    // If multiple panels are needed, you can pass the panel URL and key when instantiating the object
    public function __construct($panel_url = null, $api_key = null)
    {
        if ($panel_url) $this->PANEL_URL = $panel_url;
        if ($api_key) $this->API_KEY = $api_key;
    }

    // Example method to get panel logs
    public function getLogs()
    {
        // Construct URL
        $url = $this->PANEL_URL . '/data?action=getData';

        // Prepare POST data
        $post_data = $this->getKeyData(); // Get signature
        $post_data['table'] = 'logs';
        $post_data['limit'] = 10;
        $post_data['tojs'] = 'test';

        // Send request to the panel API
        $result = $this->httpPost($url, $post_data);

        // Parse JSON data
        $data = json_decode($result, true);
        return $data;
    }

    /**
     * Get list of websites with their IDs
     *
     * @return array List of sites with their metadata (including ID)
     */
    public function getWebsiteList($search = "", $limit = 10)
    {
        $url = $this->PANEL_URL . '/data?action=getData&table=sites';
        $post_data = $this->getKeyData();
        $post_data['limit'] = $limit;
        $post_data['search'] = $search;

        $result = $this->httpPost($url, $post_data);
        $data = json_decode($result, true);


        return $data ?? []; // 'data' contains list of websites
    }


    // Method to create a new website
    public function createWebsite($domain, $path, $userdb, $passdb, $type = 0)
    {
        $url = $this->PANEL_URL . '/site?action=AddSite';

        // Prepare POST data
        $post_data = $this->getKeyData();
        $post_data['webname'] = json_encode(['domain' => $domain, 'domainlist' => [], 'count' => 0]);
        $post_data['path'] = "/www/wwwroot/{$path}";
        $post_data['type_id'] = $type;
        $post_data['type'] = "PHP";
        $post_data['version'] = "80";
        $post_data['port'] = "80";
        $post_data['ps'] = env('APP_DOMAIN', 'interaksi.live');
        $post_data['ftp'] = "false";
        $post_data['ftp_username'] = $userdb;
        $post_data['ftp_password'] = $passdb;
        $post_data['sql'] = "true";
        $post_data['codeing'] = "utf8mb4";
        $post_data['datauser'] = $userdb;
        $post_data['datapassword'] = $passdb;

        $post_data['set_ssl'] = 1;
        // $post_data['codeing'] = "utf-8";

        // Send request to the panel API
        $result = $this->httpPost($url, $post_data);
        // Parse JSON data
        $data = json_decode($result, true);
        Log::info('createWebsite', $data);

        if (isset($data['siteStatus']) && $data['siteStatus'] === true) {
            if (isset($data['databaseStatus']) && $data['databaseStatus'] !== true) {
                throw new \Exception('Database creation failed');
            }
            return [
                'site' =>  true,
                'ssl' => isset($data['ssl']) && $data['ssl'] === true,
            ];
        } elseif (isset($data['status']) && $data['status'] === false) {
            // Log the failure message if needed
            error_log("Website creation failed: " . $data['msg']);
            throw new \Exception('Website creation failed');
        }

        throw new \Exception('Website creation failed');
    }

    /**
     * Stop a website by ID and domain name
     *
     * @param int $id Website ID
     * @param string $name Website domain name (e.g., "w2.hao.com")
     * @return array Response from API
     * @throws \Exception on failure
     */
    public function stopWebsite($id, $name)
    {
        $url = $this->PANEL_URL . '/site?action=SiteStop';
        $post_data = $this->getKeyData();
        $post_data['id'] = $id;
        $post_data['name'] = $name;

        $result = $this->httpPost($url, $post_data);
        $data = json_decode($result, true);
        Log::info('stopWebsite', $data);

        if (isset($data['status']) && $data['status'] === true) {
            return $data;
        }

        throw new \Exception('Failed to stop website: ' . ($data['msg'] ?? 'Unknown error'));
    }

    /**
     * Delete a website by ID and name, optionally remove FTP, DB, and files
     *
     * @param int $id Website ID
     * @param string $webname Site name (e.g., "w2_hao_com")
     * @param bool $deleteFtp Whether to delete FTP
     * @param bool $deleteDb Whether to delete DB
     * @param bool $deletePath Whether to delete website root directory
     * @return array Response from API
     * @throws \Exception on failure
     */
    public function deleteWebsite($id, $webname, $deleteFtp = false, $deleteDb = false, $deletePath = false)
    {
        $url = $this->PANEL_URL . '/site?action=DeleteSite';
        $post_data = $this->getKeyData();
        $post_data['id'] = $id;
        $post_data['webname'] = $webname;

        if ($deleteFtp) $post_data['ftp'] = 1;
        if ($deleteDb) $post_data['database'] = 1;
        if ($deletePath) $post_data['path'] = 1;

        $result = $this->httpPost($url, $post_data);
        $data = json_decode($result, true);
        Log::info('deleteWebsite', $data);

        if (isset($data['status']) && $data['status'] === true) {
            return $data;
        }

        throw new \Exception('Failed to delete website: ' . ($data['msg'] ?? 'Unknown error'));
    }


    /**
     * Generate an associative array with a signature
     */
    private function getKeyData()
    {
        $now_time = time();
        $post_data = array(
            'request_token' => md5($now_time . '' . md5($this->API_KEY)),
            'request_time' => $now_time
        );
        return $post_data;
    }

    /**
     * Send a POST request
     * @param String $url Target URL with http:// or https://
     * @param Array|String $data Data to be submitted
     * @param int $timeout Timeout period
     * @return string
     */
    private function httpPost($url, $data, $timeout = 60)
    {
        // Define cookie storage location
        $cookie_file = './' . md5($this->PANEL_URL) . '.cookie';
        if (!file_exists($cookie_file)) {
            $fp = fopen($cookie_file, 'w+');
            fclose($fp);
        }

        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, $url);
        curl_setopt($ch, CURLOPT_TIMEOUT, $timeout);
        curl_setopt($ch, CURLOPT_POST, 1);
        curl_setopt($ch, CURLOPT_POSTFIELDS, $data);
        curl_setopt($ch, CURLOPT_COOKIEJAR, $cookie_file);
        curl_setopt($ch, CURLOPT_COOKIEFILE, $cookie_file);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, 1);
        curl_setopt($ch, CURLOPT_HEADER, 0);
        curl_setopt($ch, CURLOPT_SSL_VERIFYHOST, false);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);

        $response = curl_exec($ch);


        if (curl_errno($ch)) {

            return 'Curl error: ' . curl_error($ch);
        }
        curl_close($ch);


        return $response;
    }



    public function getFileContent($path)
    {
        $url = $this->PANEL_URL . '/files?action=GetFileBody';
        $post_data = $this->getKeyData();
        $post_data['path'] = $path;

        $result = $this->httpPost($url, $post_data);

        $data = json_decode($result, true);

        if ($data && $data['data']) {
            return $data['data'];
        }
        return false;
    }


    public function saveFileContent($path, $content, $encoding = 'utf-8')
    {
        $url = $this->PANEL_URL . '/files?action=SaveFileBody';
        $post_data = $this->getKeyData();
        $post_data['path'] = $path;
        $post_data['data'] = $content;
        $post_data['encoding'] = $encoding;

        $result = $this->httpPost($url, $post_data);

        $data = json_decode($result, true);

        return $data && isset($data['status']) && $data['status'] === true;
    }

    public function addHeadersConfigToConfig($domain)
    {
        $configPath = "/www/server/panel/vhost/nginx/" . $domain . ".conf";
        $configContent = $this->getFileContent($configPath);

        if ($configContent === false) {
            throw new \Exception("Gagal ambil config di path: {$configPath}");
        }

        $headersToAdd =
            "    add_header 'Access-Control-Allow-Origin' '*';\n" .
            "    add_header 'Access-Control-Allow-Methods' 'GET, POST, OPTIONS';\n" .
            "    add_header 'Access-Control-Allow-Headers' 'Origin, Authorization, Content-Type, X-Requested-With';\n";


        $lines = explode("\n", $configContent);
        $newLines = [];
        foreach ($lines as $line) {
            $newLines[] = $line;
            if (preg_match('/^\s*root\s+.+;$/', $line)) {
                $newLines[] = rtrim($headersToAdd); // tambahkan header setelah baris root
            }
        }
        $newConfigContent = implode("\n", $newLines);

        $success = $this->saveFileContent($configPath, $newConfigContent);
        if (!$success) {
            throw new \Exception("Gagal menyimpan config ke path: {$configPath}");
        }

        return true;
    }
}
